function y = locswtch(u);
%----------------------------------------------------------------
% LOCSWTCH.m contains the switch-criterion for switching from
% Localizer-Armed to Localizer-Coupled.
%
% Inputsignal
% ===========
% u = [Gamma_loc d(Gamma_loc)/dt locswtch_feedback]'
%
%     Gamma_loc is the angle between the centerline of the run-
%     way and the ground-projection of the line from the air-
%     craft's c.g. to the localizer transmitter [rad].
%
%     d(Gamma_loc)/dt is the rate of change of Gamma_loc [rad/s].
%
%     locswtch_feedback is the previous output from locswtch,
%     used to determine if the LOC mode has been switched from
%     'Armed' to 'Coupled' already.
%
% Outputsignal
% ============
% y = flag variable, which determines if the Localizer mode is
%     Armed (y = 1) or Coupled (y = -1)
%----------------------------------------------------------------

% Check switch-criterion only if the NAV-mode is still in 'Armed' phase.
% ----------------------------------------------------------------------
if u(3)==1

   % Evaluate switch criterion for Localizer mode. This criterion depends
   % upon a combination of the angle Gamma_loc and its rate-of-change.
   % See refs.[1] and [2] for more details on this criterion.
   % --------------------------------------------------------------------
   if ((10*u(2)+u(1))>0.0001 & u(1)<0) | ((10*u(2)+u(1))<-0.0001 & u(1)>0)
      y = -1; % switch to Coupled
   else
      y =  1; % remain Armed
   end

else
   y = -1; % stay Coupled
end

% References
% ==========
% [1]  M.O. Rauw, A SIMULINK environment for Flight Dynamics and
%      Control analysis - Application to the DHC-2 'Beaver',
%      PART II! MSc-thesis, Delft University of Technology, Faculty 
%      of Aerospace Engineering, Delft, 1993.
%
% [2]  P.N.H. Wever, Ontwerp en implementatie van de regelwetten
%      van de De Havilland DHC-2 'Beaver' (in Dutch). MSc-thesis,
%      Delft University of Technology, Faculty of Aerospace 
%      Engineering, Delft, 1993.

%-----------------------------------------------------------------------
% The Flight Dynamics and Control Toolbox version 1.4.0. 
% (c) Copyright Marc Rauw, 1994-2005. Licensed under the Open Software 
% License version 2.1; see COPYING.TXT and LICENSE.TXT for more details.
% Last revision of this file: December 31, 2004. 
%-----------------------------------------------------------------------